﻿#region Użycie dyrektyw

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Runtime.Serialization;
using System.Text;

#endregion

namespace Chapter_17
{
    #region Program demonstrujący użycie ograniczenia szablonu

    // klienci

    public class Customer : ISerializable
    {
        #region Elementy ISerializable

        public void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            throw new NotImplementedException();
        }

        #endregion
    }

    public class LargBiz : Customer { }
    public class SmallBiz : Customer { }
    public class HomeUser : Customer { }

    // zamówienia

    public class Order : ISerializable
    {
        #region Elementy ISerializable

        public void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            throw new NotImplementedException();
        }

        #endregion
    }

    public class MailOrder : Order { }
    public class RetailOrder : Order { }

    /// <summary>
    /// Prezentuje użycie ograniczenia dla typu
    /// </summary>
    /// <typeparam name="V">Customer i ISerializable</typeparam>
    /// <typeparam name="W">Order i ISerializable</typeparam>
    public class CustomerProcessor<V, W>
        where V : Customer, ISerializable
        where W : Order, ISerializable
    {
        public void ProcessOrder(V customer, W order)
        {
            if (!ValidateAddress(customer)) return;
            if (!ValidateCredit(customer)) return;
            if (!ValidateAvailability(order)) return;

            PlaceOrder(customer, order);

            Console.WriteLine("Zamówienie wykonane!");
        }

        protected virtual bool ValidateAddress(V customer)
        {
            // sprawdzenie poprawności adresu
            return true;
        }

        protected virtual bool ValidateCredit(V customer)
        {
            // sprawdzenie poziomu zaufania dla klienta
            return true;
        }


        protected virtual bool ValidateAvailability(W order)
        {
            // sprawdzenie dostępności zamówionego towaru
            return true;
        }

        protected virtual void PlaceOrder(V customer, W order)
        {
            // zrealizowanie zamówienia
        }
    }

    #endregion

    /// <summary>
    /// Obsługa różnych typów ograniczeń
    /// </summary>
    public class ConstraintDemo
    {
        public void DoSomething<T>(T parm)
        {
        }

        /// <summary>
        /// Typ, który nie implementuje IEnumerable<T>.
        /// Używany dla celów testowych z TestInterfaceConstraints
        /// </summary>
        /// <typeparam name="T">Typ do zapamiętania</typeparam>
        class MyType<T> : IEnumerable
        {
            // wewnętrzy zasób pamięci
            private List<T> myList = new List<T>();

            // dodanie elementu do listy
            public void Add(T item)
            {
                myList.Add(item);
            }

            // dostęp do listy poprzez indeks
            public T this[int index]
            {
                get
                {
                    return myList[index];
                }
                set
                {
                    myList[index] = value;
                }
            }

            #region Elementy IEnumerable

            // opakowuje myList.GetEnumerator 
            public IEnumerator GetEnumerator()
            {
                return myList.GetEnumerator();
            }

            #endregion
        }

        /// <summary>
        /// Algorytm prezentujący i testujący
        /// ograniczenia interfejsu
        /// </summary>
        public void TestInterfaceConstraints()
        {
            // lista typów wartościowych jest poprawna
            List<int> myInts = new List<int>();
            myInts.AddRange(new int[] { 1, 2, 3, 4, 5, 6 });

            PrintEnumerableValues<List<int>, int>(myInts);

            // lista typów referencyjnych jest poprawna
            List<string> myStrings = new List<string>();
            myStrings.AddRange(new string[] { "jeden", "dwa", "trzy" });

            PrintEnumerableValues<List<string>, string>(myStrings);

            // tablica liczb typu double jest poprawna
            double[] myDoubles = { 7.7 };

            PrintEnumerableValues<double[], double>(myDoubles);

            // typ stworzony przez użytkownika używający indeksatora nie bedzie działać z PrintEnumerableValues
            // ponieważ nie implementuje ograniczenia IEnumerable<T>
            MyType<decimal> myType = new MyType<decimal>();
            myType.Add(9.9m);

            // nie można skompilować, ponieważ MyType nie jest interfejsem IEnumerable<decimal>,
            // który jest ograniczeniem metody PrintEnumerableValues
            //PrintEnumerableValues<MyType<decimal>, decimal>(myType);
        }

        /// <summary>
        /// Wyświetla tylko te elementy kolekcji,
        /// które implementują IEnumerable<T>"/>
        /// </summary>
        /// <typeparam name="T">Ograniczony jako IEnumerable<U></typeparam>
        /// <typeparam name="U">Typ kolekcji, który może być przetwarzany</typeparam>
        /// <param name="myItems">Kolekcja typu T</param>
        private void PrintEnumerableValues<T, U>(T myItems)
            where T : IEnumerable<U>
        {
            foreach (U item in myItems)
            {
                Console.WriteLine(item);
            }
        }

        /// <summary>
        /// Algorytm prezentujący i testujący
        /// ograniczenia interfejsu
        /// </summary>
        public void TestBaseClassConstraints()
        {
            byte[] bytes = Encoding.Unicode.GetBytes(
                "Zwinny brązowy lis ponownie przeskoczył przez leniwego psa");

            MemoryStream memStream = new MemoryStream(bytes);

            // MemoryStream pochodzi z klasy Stream - ta konstrukcja jest prawidłowa
            PrintStreamValues<MemoryStream>(memStream);

            // StringReader nie jest klasą Stream - konstrukcja nie jest prawidłowa
            StringReader strRdr = new StringReader("Jestem elementem StringReader!");

            // błąd czasu kompilacji
            //PrintStreamValues<StringReader>(strRdr);
        }

        /// <summary>
        /// Kod będzie działać z dowolnym typem pochodzącym z klasy Stream
        /// </summary>
        /// <typeparam name="T">Musi być strumieniem</typeparam>
        /// <param name="stream">Źródło odczytywanych danych</param>
        private void PrintStreamValues<T>(T stream)
            where T : Stream
        {
            int ByteSize = 1024;
            int count = 0;

            StringBuilder result = new StringBuilder();

            byte[] bytes = new byte[ByteSize];

            do
            {
                count = stream.Read(bytes, 0, ByteSize);

                // zamień na łańcuch bez względu na format
                result.Append(
                    Encoding.Unicode.GetString(bytes).
                    Substring(0, count - 1));
            }
            while (count == ByteSize);

            Console.WriteLine(result.ToString());
        }

        /// <summary>
        /// Przykładowe ograniczenia typu referencyjnego i wartościowego
        /// </summary>
        public void TestReferenceAndValueTypeConstraints()
        {
            MyRefType refType = new MyRefType();
            refType.Val = 3;

            // modyfikuje lokalny typ referencyjny
            CheckForReferenceType<MyRefType>(refType);

            Console.WriteLine(
                "refType.Val was 3 and is now " + refType.Val);

            MyValType valType = new MyValType();
            valType.Val = 3;

            // nie zmienia lokalnej wartości
            CheckForValueType<MyValType>(valType);

            Console.WriteLine(
                "valType.Val was 3 and is now " + valType.Val);

            // błędy czasu kompilacji

            // narusza ograniczenie typu referencyjnego
            //			CheckForReferenceType<MyValType>(valType);

            // narusza ograniczenie typu wartościowego
            //			CheckForValueType<MyRefType>(refType);
        }

        /// <summary>
        /// Ograniczenie metody, która prezentuje działanie mechanizmu ograniczeń
        /// dla typu referencyjnego i wartościowego
        /// </summary>
        private interface IHoldVal
        {
            int Val
            {
                get;
                set;
            }
        }

        /// <summary>
        /// Przykład klasy obsługującej 
        /// ograniczenia typu referencyjnego
        /// oraz ograniczenie konstruktora
        /// </summary>
        public class MyRefType : IHoldVal
        {
            // musi używać domyślnego konstruktora
            public MyRefType() { }

            public MyRefType(int myParam)
            {
                val = myParam;
            }

            private int val;

            // implementuje IHoldVal
            public int Val
            {
                get { return val; }
                set { val = value; }
            }
        }

        /// <summary>
        /// Przykładowa klasa dla programu demonstracyjnego
        /// prezentującego użycie ograniczenia typu wartościowego
        /// oraz ograniczenia konstruktora
        /// </summary>
        public struct MyValType : IHoldVal
        {
            public MyValType(int myParam)
            {
                val = myParam;
            }

            private int val;

            // implementuje IHoldVal
            public int Val
            {
                get { return val; }
                set { val = value; }
            }
        }


        /// <summary>
        /// Prezentuje użycie ograniczenia typu referencyjnego
        /// </summary>
        /// <typeparam name="T">Parametr typu</typeparam>
        /// <param name="type">Parametr wejściowy</param>
        private void CheckForReferenceType<T>(T type)
            where T : class, IHoldVal
        {
            type.Val = 5;
        }

        /// <summary>
        /// Prezentuje użycie ograniczenia typu wartościowego
        /// </summary>
        /// <typeparam name="T">Parametr typu</typeparam>
        /// <param name="type">Parametr wejściowy</param>
        private void CheckForValueType<T>(T type)
            where T : struct, IHoldVal
        {
            type.Val = 5;
        }

        /// <summary>
        /// Prezentuje działanie ograniczenia konstruktora
        /// </summary>
        public void TestConstructorConstraints()
        {
            // działa z typami referencyjnymi
            IHoldVal myRef = CreateNewInstance<MyRefType>();
            Console.WriteLine(myRef.Val);

            // działa z typami wartościowymi
            IHoldVal myVal = CreateNewInstance<MyValType>();
            Console.WriteLine(myVal.Val);
        }

        /// <summary>
        /// Tworzy nową instancję obiektu
        /// </summary>
        /// <typeparam name="T">Typ konkretyzowany</typeparam>
        /// <returns>Nowa instancja</returns>
        private T CreateNewInstance<T>()
            where T : IHoldVal, new()
        {
            T myNewType = new T();
            myNewType.Val = 7;

            // użycie ograniczenia konstruktora jest możliwe wyłącznie
            // dla domyślnych konstruktorów -
            // poniższy kod powoduje powstanie błędu czasu kompilacji
            //			T anotherNewType = new T(7);

            return myNewType;
        }

        //private T CreateNewInstance<T>()
        //where T : IDisposable, new()
        //{
        //    T myNewType = new T();
        //    //myNewType.Val = 7;

        //    // użycie ograniczenia konstruktora jest możliwe wyłącznie
        //    // dla domyślnych konstruktorów -
        //    // poniższy kod powoduje powstanie błędu czasu kompilacji
        //    //			T anotherNewType = new T(7);

        //    return myNewType;
        //}

    }

}